var MemoryAnalyzer;
(function (MemoryAnalyzer) {
    var SnapshotFilterCache = (function () {
        function SnapshotFilterCache(data, filterApplied, sorterApplied, functionName) {
            this._originalData = data;
            this._appliedFilter = filterApplied;
            this._appliedFunctionName = functionName;
            this._appliedSortProperty = sorterApplied.sortProperty;
            this._appliedSortOrder = sorterApplied.sortOrder;
            this._originalData = this._originalData.sort(sorterApplied.sortComparer);
        }
        SnapshotFilterCache.isItemExcludedByFilter = function isItemExcludedByFilter(filter, item, prop) {
            if(filter.isChangedOnly && item && !(item.isChanged || item.isRelatedToChanged)) {
                return true;
            }
            if(filter.isFlatView && item && item.isRelatedToChanged) {
                return true;
            }
            if(!filter.showUnknownSizes && item && !(item.size) && !(item.retainedSize)) {
                return true;
            }
            if(!filter.showUnknownTypes && item && item.type === "Unknown") {
                return true;
            }
            if(!filter.showBuiltIns && ((prop && prop.isBuiltIn) || (item && item.isBuiltIn))) {
                return true;
            }
            return false;
        }
        SnapshotFilterCache.createRegexFromFilter = function createRegexFromFilter(filter) {
            if(filter.filterString) {
                var regexInput = filter.filterString.replace(/[-{}()[\]*.+,^?$|#\s\\]/g, "\\$&");
                return new RegExp(regexInput, "i");
            }
            return null;
        }
        SnapshotFilterCache.prototype.canUseCacheWithFunction = function (functionName) {
            if(this._originalData && this._appliedFilter && this._appliedFunctionName === functionName) {
                return true;
            }
            return false;
        };
        SnapshotFilterCache.prototype.getFilteredCache = function (filter, sorter) {
            var ret = [];
            var filterRegEx = SnapshotFilterCache.createRegexFromFilter(filter);
            var cache = this._originalData;
            if(filter.filterString && this._latestFilterString && this._appliedFilter.showBuiltIns === filter.showBuiltIns && this._appliedFilter.showUnknownTypes === filter.showUnknownTypes && this._appliedFilter.showUnknownSizes === filter.showUnknownSizes && this._latestFilterString.length <= filter.filterString.length && filter.filterString.substr(0, this._latestFilterString.length) === this._latestFilterString) {
                cache = this._latestData;
            }
            if(this._appliedSortProperty !== sorter.sortProperty) {
                this._originalData.sort(sorter.sortComparer);
                if(cache !== this._originalData) {
                    cache.sort(sorter.sortComparer);
                }
            } else {
                if(this._appliedSortOrder !== sorter.sortOrder) {
                    Array.stableReverse(this._originalData, sorter.sortComparer);
                    if(cache !== this._originalData) {
                        Array.stableReverse(cache, sorter.sortComparer);
                    }
                }
            }
            this._appliedSortProperty = sorter.sortProperty;
            this._appliedSortOrder = sorter.sortOrder;
            for(var i = 0; i < cache.length; i++) {
                if(filterRegEx && !filterRegEx.test(cache[i].name)) {
                    continue;
                }
                if(SnapshotFilterCache.isItemExcludedByFilter(filter, cache[i])) {
                    continue;
                }
                ret.push(cache[i]);
            }
            this._latestFilterString = filter.filterString;
            this._latestData = ret;
            return ret;
        };
        return SnapshotFilterCache;
    })();
    MemoryAnalyzer.SnapshotFilterCache = SnapshotFilterCache;    
    var Snapshot = (function () {
        function Snapshot(heapObjectMap, graph, pointerSize, isDiff) {
            this._heapObjectMap = heapObjectMap;
            this._graph = graph;
            this._pointerSize = pointerSize;
            this._isDiff = isDiff;
        }
        Object.defineProperty(Snapshot.prototype, "heapObjectMap", {
            get: function () {
                return this._heapObjectMap;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Snapshot.prototype, "graph", {
            get: function () {
                return this._graph;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Snapshot.prototype, "isDiff", {
            get: function () {
                return this._isDiff;
            },
            enumerable: true,
            configurable: true
        });
        Snapshot.prototype.getHeapObject = function (objectId) {
            return this._heapObjectMap[objectId];
        };
        Snapshot.prototype.hasHeapObject = function (objectId) {
            return this._heapObjectMap.hasOwnProperty(objectId);
        };
        Snapshot.prototype.getAllObjects = function (filter, sorter) {
            var functionId = "getAllObjects[" + filter.displayObjectIDs + "]";
            if(!this._cache || !this._cache.canUseCacheWithFunction(functionId)) {
                var ret = [];
                var objects = MemoryAnalyzer.HeapObjectHelpers.getHeapObjectsFromMap(this._heapObjectMap);
                for(var i = 0; i < objects.length; i++) {
                    var heapObject = objects[i];
                    var namedHeapObject = new MemoryAnalyzer.NamedHeapObject(heapObject, this.hasChildrenWhenFiltered(heapObject, filter), filter.displayObjectIDs);
                    ret.push(namedHeapObject);
                }
                this._cache = new SnapshotFilterCache(ret, filter, sorter, functionId);
            }
            return this._cache.getFilteredCache(filter, sorter);
        };
        Snapshot.prototype.getAllRootObjects = function (filter, sorter) {
            var functionId = "getAllRootObjects[" + filter.displayObjectIDs + "]";
            if(!this._cache || !this._cache.canUseCacheWithFunction(functionId)) {
                var ret = [];
                var ids = this._graph.roots;
                for(var i = 0; i < ids.length; i++) {
                    var heapObject = this.getHeapObject(ids[i]);
                    if(heapObject) {
                        var namedHeapObject = new MemoryAnalyzer.NamedHeapObject(heapObject, this.hasChildrenWhenFiltered(heapObject, filter), filter.displayObjectIDs);
                        ret.push(namedHeapObject);
                    }
                }
                this._cache = new SnapshotFilterCache(ret, filter, sorter, functionId);
            }
            return this._cache.getFilteredCache(filter, sorter);
        };
        Snapshot.prototype.getAllTypes = function (filter, sorter) {
            var functionId = "getAllTypes";
            functionId += "[" + [
                filter.isChangedOnly, 
                filter.isFlatView, 
                filter.showBuiltIns, 
                filter.showUnknownSizes, 
                filter.showUnknownTypes
            ].join(",") + "]";
            if(!this._cache || !this._cache.canUseCacheWithFunction(functionId)) {
                var ret = [];
                var types = this._graph.types;
                var countFilter = {
                    isChangedOnly: false,
                    isFlatView: false,
                    showBuiltIns: filter.showBuiltIns,
                    showUnknownTypes: filter.showUnknownTypes,
                    showUnknownSizes: filter.showUnknownSizes,
                    filterString: filter.filterString,
                    displayObjectIDs: filter.displayObjectIDs
                };
                for(var typeName in types) {
                    if(types.hasOwnProperty(typeName)) {
                        var typeItem = types[typeName];
                        if(typeItem) {
                            var filteredChildrenCount = this.getGroupFilteredChildCount(typeItem, countFilter);
                            if(filteredChildrenCount.childrenCount > 0) {
                                var treeItem = new MemoryAnalyzer.TypeDataObject(typeName, typeItem, filteredChildrenCount.childrenCount, filteredChildrenCount.childrenCountDiff);
                                ret.push(treeItem);
                            }
                        }
                    }
                }
                this._cache = new SnapshotFilterCache(ret, filter, sorter, functionId);
            }
            return this._cache.getFilteredCache(filter, sorter);
        };
        Snapshot.prototype.getAllMarkupObjects = function (filter, sorter) {
            var functionId = "getAllMarkupObjects[" + filter.displayObjectIDs + "]";
            if(!this._cache || !this._cache.canUseCacheWithFunction(functionId)) {
                var ret = [];
                var ids = this._graph.rootMarkups;
                for(var i = 0; i < ids.length; i++) {
                    var heapObject = this.getHeapObject(ids[i]);
                    if(heapObject) {
                        var namedHeapObject = new MemoryAnalyzer.NamedHeapObject(heapObject, this.hasChildrenWhenFiltered(heapObject, filter), filter.displayObjectIDs);
                        ret.push(namedHeapObject);
                    }
                }
                this._cache = new SnapshotFilterCache(ret, filter, sorter, functionId);
            }
            return this._cache.getFilteredCache(filter, sorter);
        };
        Snapshot.prototype.getAllWinRTObjects = function (filter, sorter) {
            var functionId = "getAllWinRTObjects[" + filter.displayObjectIDs + "]";
            if(!this._cache || !this._cache.canUseCacheWithFunction(functionId)) {
                var ret = [];
                var ids = this._graph.winRTs;
                for(var i = 0; i < ids.length; i++) {
                    var heapObject = this.getHeapObject(ids[i]);
                    if(heapObject) {
                        var namedHeapObject = new MemoryAnalyzer.NamedHeapObject(heapObject, this.hasChildrenWhenFiltered(heapObject, filter), filter.displayObjectIDs);
                        ret.push(namedHeapObject);
                    }
                }
                this._cache = new SnapshotFilterCache(ret, filter, sorter, functionId);
            }
            return this._cache.getFilteredCache(filter, sorter);
        };
        Snapshot.prototype.getAllChildrenForObjectOrType = function (id, filter, sorter) {
            var objectId = parseInt(id);
            if(!isNaN(objectId)) {
                return this.getAllChildrenForObject(objectId, filter, sorter);
            } else {
                return this.getAllObjectsByType(id, filter, sorter);
            }
        };
        Snapshot.prototype.getAllObjectsByType = function (typeName, filter, sorter) {
            var types = this._graph.types;
            var type = types[typeName];
            if(!type) {
                return [];
            }
            var ret = [];
            for(var i = 0; type.objectIds && i < type.objectIds.length; i++) {
                var heapObject = this.getHeapObject(type.objectIds[i]);
                if(heapObject && !SnapshotFilterCache.isItemExcludedByFilter(filter, heapObject)) {
                    var namedHeapObject = new MemoryAnalyzer.NamedHeapObject(heapObject, this.hasChildrenWhenFiltered(heapObject, filter), filter.displayObjectIDs);
                    ret.push(namedHeapObject);
                }
            }
            var sortedRet = ret.sort(sorter.sortComparer);
            return sortedRet;
        };
        Snapshot.prototype.getAllChildrenForObject = function (objectId, filter, sorter) {
            var ret = [];
            var heapObject;
            var instanceName;
            var treeItem;
            var parent = this.getHeapObject(objectId);
            if(!parent) {
                return ret;
            }
            var directChildren = [];
            var nameOverrides = [];
            var childrenCount = MemoryAnalyzer.HeapObjectHelpers.getDirectChildren(parent, this._heapObjectMap, MemoryAnalyzer.DirectChildrenKind.All, directChildren, nameOverrides);
            for(var i = 0; i < childrenCount; i++) {
                var prop = directChildren[i];
                var nameOverride = nameOverrides[i];
                if(prop.objectId) {
                    heapObject = this.getHeapObject(prop.objectId);
                } else {
                    heapObject = new MemoryAnalyzer.ValueHeapObject(parent, prop, this._pointerSize);
                }
                if(!SnapshotFilterCache.isItemExcludedByFilter(filter, heapObject, prop)) {
                    if(!nameOverride) {
                        nameOverride = prop.name;
                    }
                    treeItem = new MemoryAnalyzer.NamedHeapObject(heapObject, this.hasChildrenWhenFiltered(heapObject, filter), filter.displayObjectIDs, nameOverride);
                    ret.push(treeItem);
                }
            }
            var sortedRet = ret.sort(sorter.sortComparer);
            return sortedRet;
        };
        Snapshot.prototype.getReferenceTreeItems = function (objectId, sorter, displayObjectIds) {
            var result = [];
            var obj = this.getHeapObject(objectId);
            if(!obj || ((!obj.parents || obj.parents.length === 0) && !obj.size && obj.type === "Unknown")) {
                return result;
            }
            var parentIds = obj.parents;
            if(!parentIds) {
                result.push(new MemoryAnalyzer.ReferenceDataObject(obj, displayObjectIds));
            } else {
                for(var i = 0; i < parentIds.length; i++) {
                    var parent = this.getHeapObject(parentIds[i]);
                    var childrenProps = [];
                    var overrideNames = [];
                    var childrenCount = MemoryAnalyzer.HeapObjectHelpers.getDirectChildren(parent, this._heapObjectMap, MemoryAnalyzer.DirectChildrenKind.All, childrenProps, overrideNames);
                    for(var childIndex = 0; childIndex < childrenCount; childIndex++) {
                        var prop = childrenProps[childIndex];
                        if(prop.objectId === objectId) {
                            var overrideName = overrideNames[childIndex];
                            if(!overrideName) {
                                overrideName = prop.name;
                            }
                            result.push(new MemoryAnalyzer.ReferenceDataObject(obj, displayObjectIds, parent.objectId, overrideName, ((parent.size ? true : false) || parent.type !== "Unknown")));
                        }
                    }
                }
            }
            result.sort(sorter.sortComparer);
            return result;
        };
        Snapshot.prototype.filterIncludeCallback = function (filter, prop) {
            return !SnapshotFilterCache.isItemExcludedByFilter(filter, this.getHeapObject(prop.objectId), prop);
        };
        Snapshot.prototype.hasChildrenWhenFiltered = function (obj, filter) {
            var directChildren = [];
            return MemoryAnalyzer.HeapObjectHelpers.checkForChildrenWhenFiltered(obj, this._heapObjectMap, this.filterIncludeCallback.bind(this, filter));
        };
        Snapshot.prototype.getGroupFilteredChildCount = function (typeGroup, filter) {
            var count = 0;
            var countDiff = 0;
            var members = typeGroup.objectIds;
            for(var i = 0; i < members.length; i++) {
                var object = (this.getHeapObject(members[i]));
                if(!SnapshotFilterCache.isItemExcludedByFilter(filter, object)) {
                    count++;
                    if(object.isNew) {
                        countDiff++;
                    }
                }
            }
            return {
                childrenCount: count,
                childrenCountDiff: countDiff
            };
        };
        Snapshot.prototype.getPathToRoot = function (objectId, filter, parentObjectId) {
            if (typeof parentObjectId === "undefined") { parentObjectId = -1; }
            var ancestorsMap = {
            };
            ancestorsMap[objectId] = [
                objectId
            ];
            var visited = {
            };
            var queue = [];
            if(parentObjectId >= 0) {
                var isValidParent = false;
                var startingParents = this.getHeapObject(objectId).parents;
                for(var i = 0; startingParents && i < startingParents.length; i++) {
                    if(startingParents[i] === parentObjectId) {
                        isValidParent = true;
                        break;
                    }
                }
                if(isValidParent) {
                    ancestorsMap[parentObjectId] = [
                        objectId, 
                        parentObjectId
                    ];
                    visited[parentObjectId] = true;
                    queue.push(parentObjectId);
                }
            } else {
                visited[objectId] = true;
                queue.push(objectId);
            }
            var foundRootId;
            while(queue.length > 0) {
                var currentId = queue.shift();
                var currentObj = this.getHeapObject(currentId);
                if(currentObj && !SnapshotFilterCache.isItemExcludedByFilter(filter, currentObj)) {
                    if(currentObj.isRoot) {
                        foundRootId = currentId;
                        break;
                    }
                    var directParents = currentObj.parents;
                    for(var i = 0; directParents && i < directParents.length; i++) {
                        var parentId = directParents[i];
                        if(typeof parentId !== "undefined") {
                            if(!visited[parentId]) {
                                visited[parentId] = true;
                                queue.push(parentId);
                                ancestorsMap[parentId] = ancestorsMap[currentId].slice(0);
                                ancestorsMap[parentId].push(parentId);
                            }
                        }
                    }
                }
            }
            if(foundRootId) {
                return ancestorsMap[foundRootId];
            }
            return [];
        };
        return Snapshot;
    })();
    MemoryAnalyzer.Snapshot = Snapshot;    
})(MemoryAnalyzer || (MemoryAnalyzer = {}));
//@ sourceMappingURL=snapshot.js.map

// SIG // Begin signature block
// SIG // MIIajQYJKoZIhvcNAQcCoIIafjCCGnoCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFLP0Stc5buLv
// SIG // aWNns7e/Eo4No8VRoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // kkoAAAAAACAwDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OVoXDTEz
// SIG // MDQwOTIyMjU1OVowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpCOEVDLTMwQTQtNzE0NDElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAM1jw/ei
// SIG // tUfZ+TmUU6xrj6Z5OCH00W49FTgWwXMsmY/74Dxb4aJM
// SIG // i7Kri7TySse5k1DRJvWHU7B6dfNHDxcrZyxk62DnSozg
// SIG // i17EVmk3OioEXRcByL+pt9PJq6ORqIHjPy232OTEeAB5
// SIG // Oc/9x2TiIxJ4ngx2J0mPmqwOdOMGVVVJyO2hfHBFYX6y
// SIG // cRYe4cFBudLSMulSJPM2UATX3W88SdUL1HZA/GVlE36V
// SIG // UTrV/7iap1drSxXlN1gf3AANxa7q34FH+fBSrubPWqzg
// SIG // FEqmcZSA+v2wIzBg6YNgrA4kHv8R8uelVWKV7p9/ninW
// SIG // zUsKdoPwQwTfBkkg8lNaRLBRejkCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBTNGaxhTZRnK/avlHVZ2/BYAIOhOjAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBRHNbfNh3cgLwCp8aZ3xbI
// SIG // kAZpFZoyufNkENKK82IpG3mPymCps13E5BYtNYxEm/H0
// SIG // XGGkQa6ai7pQ0Wp5arNijJ1NUVALqY7Uv6IQwEfVTnVS
// SIG // iR4/lmqPLkAUBnLuP3BZkl2F7YOZ+oKEnuQDASETqyfW
// SIG // zHFJ5dod/288CU7VjWboDMl/7jEUAjdfe2nsiT5FfyVE
// SIG // 5x8a1sUaw0rk4fGEmOdP+amYpxhG7IRs7KkDCv18elId
// SIG // nGukqA+YkqSSeFwreON9ssfZtnB931tzU7+q1GZQS/DJ
// SIG // O5WF5cFKZZ0lWFC7IFSReTobB1xqVyivMcef58Md7kf9
// SIG // J9d/z3TcZcU/MIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSAMIIEfAIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGiMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBScmeGifIh5bAS7b9h4XF8d
// SIG // dS7/UDBCBgorBgEEAYI3AgEMMTQwMqAYgBYAcwBuAGEA
// SIG // cABzAGgAbwB0AC4AagBzoRaAFGh0dHA6Ly9taWNyb3Nv
// SIG // ZnQuY29tMA0GCSqGSIb3DQEBAQUABIIBANx3cGU/3s9Q
// SIG // OBuOGfelOrR+PGzudyHBNuGwHy7HtUB67vFcF4VYrX0X
// SIG // 37mjzY6bqKllooCTvXgU9rTxiIQP7QYtKTxuD3uyp+h1
// SIG // fLKxs466nqLMiAF22gMTIESZxHWF0HGCCHURng2RSkZA
// SIG // /lHFeux5ZMjzfmN9htpX2++E8hh+5tz9Z4+2Nu/hNVBP
// SIG // IFFWrC6e03baSi/3T4pA0MnTBL4hoL149H57eQ88hp0F
// SIG // DSaro149Mw3p5k3NvPnt2Tq4TgGjRlAAIIhLdHjIouAZ
// SIG // hWTTGMuc0QI4xhEJwcZsOnsURGfSOjgY2gLBPKIrB87V
// SIG // 0K8Wr83m+btQDZS2oDRxhh6hggIfMIICGwYJKoZIhvcN
// SIG // AQkGMYICDDCCAggCAQEwgYUwdzELMAkGA1UEBhMCVVMx
// SIG // EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
// SIG // ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
// SIG // dGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRpbWUtU3Rh
// SIG // bXAgUENBAgphApJKAAAAAAAgMAkGBSsOAwIaBQCgXTAY
// SIG // BgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3
// SIG // DQEJBTEPFw0xMzAzMTUwNjMzNThaMCMGCSqGSIb3DQEJ
// SIG // BDEWBBQvPeq7CSyca5PdBAcrb2bS2ORRtTANBgkqhkiG
// SIG // 9w0BAQUFAASCAQB1USIVSYqwfkHDkFzDktgcZ+YMJRne
// SIG // nNYksbu6eC2EtBeD6RZLsNtbs0XDZcNo5Ey5pMpLiy26
// SIG // 1GVKsqVEBtwyeDU9cAP+88IWLx1wWOCjx4CIa0fH9rJc
// SIG // HnzZt28Vip3znVwphZbGKXBnkOrYPqUhncSa0oT8TIjT
// SIG // GuQMk01ps1Xd/jyxe7w2uet0iRlMUaOu8dyJwRfep4Rv
// SIG // qjMeQVMRN+xn0p6SeKUHyBRoqO3IQFFqlMcs116H0crC
// SIG // cR0VEYUifgwICgLAlE98biCkiZJokfYB5KjT1/kB/5+A
// SIG // hOYZD3k0r45RF+uMgfZHfCTMnAUdS0UbCdq/FGBYlbmD
// SIG // IvHS
// SIG // End signature block
